#version 330 core

out vec4 FragColor;

in vec3 FragPos;
in vec3 Normal;
in vec2 TexCoord;

uniform vec3 lightPos;
uniform vec3 lightColor;
uniform vec3 objectColor;
uniform vec3 materialColor;
uniform vec3 viewPos;
uniform sampler2D diffuseTexture;
uniform bool hasTexture;
uniform bool hasMaterialColor;

void main()
{
    // Higher ambient lighting to ensure all sides are visible
    float ambientStrength = 0.4;
    vec3 ambient = ambientStrength * lightColor;
  	
    // Primary directional light
    vec3 norm = normalize(Normal);
    vec3 lightDir = normalize(lightPos - FragPos);
    float diff = max(dot(norm, lightDir), 0.0);
    vec3 diffuse = diff * lightColor * 0.6; // Reduce primary light contribution
    
    // Add secondary fill light from opposite direction for better illumination
    vec3 fillLightDir = normalize(-lightPos - FragPos);
    float fillDiff = max(dot(norm, fillLightDir), 0.0);
    vec3 fillLight = fillDiff * lightColor * 0.3; // Softer fill light
    
    // specular (reduced for more natural look)
    float specularStrength = 0.2;
    vec3 viewDir = normalize(viewPos - FragPos);
    vec3 reflectDir = reflect(-lightDir, norm);  
    float spec = pow(max(dot(viewDir, reflectDir), 0.0), 32);
    vec3 specular = specularStrength * spec * lightColor;  
    
    // Get base color from texture, material color, or default uniform
    vec3 baseColor;
    if (hasTexture)
    {
        baseColor = texture(diffuseTexture, TexCoord).rgb;
    }
    else if (hasMaterialColor)
    {
        baseColor = materialColor;
    }
    else
    {
        baseColor = objectColor;
    }
        
    vec3 result = (ambient + diffuse + fillLight + specular) * baseColor;
    FragColor = vec4(result, 1.0);
}
